import os
import logging
from datetime import datetime

# 配置日志
log_filename = f"delete_unwanted_files_{datetime.now().strftime('%Y%m%d_%H%M%S')}.log"
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler(log_filename, encoding='utf-8'),
        logging.StreamHandler() # 同时输出到控制台
    ]
)
logger = logging.getLogger(__name__)

# --- 配置: 允许保留的后缀列表 ---
# 这些后缀（大小写不敏感）如果出现在文件名中，文件将被保留。
# 否则，.txt 文件将被删除。
ALLOWED_SUFFIXES = ["c3","c4", "g1", "g2", "g3","gpt", "kimi", "grok", "deep"]

def has_any_allowed_suffix(filename_without_ext):
    """
    检查文件名（不含扩展名）是否包含任何允许的后缀（大小写不敏感）。
    如果包含，则返回 True (表示应保留文件)。
    """
    filename_lower = filename_without_ext.lower()
    for suffix in ALLOWED_SUFFIXES:
        if suffix.lower() in filename_lower:
            return True # 找到一个允许的后缀，所以保留此文件
    return False # 未找到任何允许的后缀，所以删除此文件

def delete_unwanted_txt_files(root_dir):
    """
    遍历指定目录及其子目录，删除所有不含指定后缀的txt文件。
    """
    logger.info(f"开始扫描目录: {root_dir}")
    logger.info(f"将保留文件名中包含以下任何后缀的 .txt 文件 (大小写不敏感): {', '.join(ALLOWED_SUFFIXES)}")
    logger.info("不含上述任何后缀的 .txt 文件将被删除。")

    total_files_scanned = 0
    files_deleted = 0
    files_kept = 0
    deleted_files_list = []
    kept_files_list = []
    skipped_non_txt_list = []

    for dirpath, dirnames, filenames in os.walk(root_dir):
        for filename in filenames:
            total_files_scanned += 1
            full_path = os.path.join(dirpath, filename)

            # 1. 检查是否是 .txt 文件
            if not filename.lower().endswith('.txt'):
                logger.debug(f"跳过 (非txt文件): {filename}")
                skipped_non_txt_list.append(full_path)
                continue

            # 2. 提取不带扩展名的文件名，用于后缀检查
            filename_without_ext = filename[:-4] # 移除 ".txt"

            # 3. 根据规则判断是否保留或删除
            if has_any_allowed_suffix(filename_without_ext):
                logger.info(f"保留文件 (文件名 '{filename_without_ext}' 包含允许后缀): {filename}")
                kept_files_list.append(full_path)
                files_kept += 1
            else:
                # 不含允许后缀，准备删除
                try:
                    logger.info(f"准备删除文件 (文件名 '{filename_without_ext}' 不含任何允许后缀): {filename}")
                    os.remove(full_path)
                    logger.info(f"成功删除: {full_path}")
                    deleted_files_list.append(full_path)
                    files_deleted += 1
                except OSError as e:
                    logger.error(f"删除文件 '{full_path}' 时发生操作系统错误: {e}。文件将保留。")
                    kept_files_list.append(full_path) # 删除失败，文件实际被保留
                    files_kept += 1
                except Exception as e:
                    logger.error(f"删除文件 '{full_path}' 时发生未知错误: {e}。文件将保留。")
                    kept_files_list.append(full_path) # 删除失败，文件实际被保留
                    files_kept += 1

    logger.info("\n========== 处理总结 ==========")
    logger.info(f"总共扫描文件: {total_files_scanned}")
    logger.info(f"已保留的 .txt 文件: {files_kept}")
    logger.info(f"已删除的 .txt 文件: {files_deleted}")
    logger.info(f"已跳过的非 .txt 文件: {len(skipped_non_txt_list)}")
    logger.info("================================")
    logger.info(f"详细日志已保存到: {log_filename}")

    if deleted_files_list:
        logger.info("\n--- 已删除文件列表 ---")
        for f in deleted_files_list:
            logger.info(f"- {f}")
    if kept_files_list:
        logger.info("\n--- 已保留文件列表 ---")
        for f in kept_files_list:
            logger.info(f"- {f}")
    if skipped_non_txt_list:
        logger.info("\n--- 已跳过非 .txt 文件列表 ---")
        for f in skipped_non_txt_list:
            logger.info(f"- {f}")


if __name__ == "__main__":
    root_directory = input("请输入要处理的根目录路径 (例如: D:\\MyFiles): ")
    
    if not os.path.isdir(root_directory):
        logger.error(f"输入的路径 '{root_directory}' 不是一个有效的目录。请检查。")
    else:
        delete_unwanted_txt_files(root_directory)
    
    input("\n程序运行完毕，请查看日志文件。按 Enter 键退出...")

